CREATE OR REPLACE PACKAGE sim2_web_txt_file_pkg IS

  -- Author  : KHENAS
  -- Created : 7/18/01 4:10:14 PM
  -- Purpose : replace the procedure in the impf2_d library that reads a text file into sim_imp_lines
  
  FUNCTION READ_TEXT_FILE (p_file_name IN OUT VARCHAR2, p_imp_type IN VARCHAR2, 
                           p_org_id IN OUT VARCHAR2, p_imp_cfg_seq IN NUMBER, 
                           p_import_log_seq IN OUT NUMBER) 
                           RETURN VARCHAR2;
                           
END sim2_web_txt_file_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim2_web_txt_file_pkg IS

/****************************************************************************************/
FUNCTION READ_TEXT_FILE (p_file_name IN OUT VARCHAR2, p_imp_type IN VARCHAR2, 
                         p_org_id IN OUT VARCHAR2, p_imp_cfg_seq IN NUMBER, 
                         p_import_log_seq IN OUT NUMBER) 
                         RETURN VARCHAR2 IS
                         
   /* kms (7/18/01) - copied function from impf2_d.pll and modified to run
      off the database.  Did this to insure that it is looking on the server
      for a file when the impf2_d form is being run over the web. */
   /* function to write the contents of a specified text file to a table */
   v_file_handle         UTL_FILE.FILE_TYPE;  
   v_path                VARCHAR2(40) := 'd:\ora_apps\sim2\';
   v_line_seq            NUMBER(10);
   v_line_cnt            NUMBER(10) := 0;
   v_text_line           VARCHAR2(1999);
   v_delimiter           VARCHAR2(1);
   v_no_fields           NUMBER(3);
   v_no_imp_fields       NUMBER(3);
   v_org_is_nbr          NUMBER(8);
  
   ex_bad_file_ext       EXCEPTION;  
   ex_bad_file_handle 	 EXCEPTION;
   ex_no_sequence     	 EXCEPTION;
   ex_bad_cfg         	 EXCEPTION;
   ex_wrong_delimiter 	 EXCEPTION;
   ex_wrong_no_fields 	 EXCEPTION;

   CURSOR C_GET_NEXT_LINE_SEQ IS
      SELECT SIM_IMP_LINES_SEQ.NEXTVAL FROM DUAL;

   CURSOR c_get_delimiter(p_imp_cfg_seq IN NUMBER) IS
      SELECT SICC_DELIMITER FROM SIM_IMP_CFG WHERE SICC_SEQ = p_imp_cfg_seq;
      
   CURSOR c_get_org_is IS
      SELECT TSMORGAN_IS_NUMBER FROM TSMORGAN WHERE ORG_ID = RPAD(p_org_id,8);
   
   CURSOR c_get_no_imp_fields(p_imp_cfg_seq IN NUMBER) IS
      SELECT COUNT(1) FROM SIM_IMP_CFG, SIM_IMP_CFG_DETS 
        WHERE SICDT_INCLUDED IS NOT NULL AND SICC_SEQ = p_imp_cfg_seq 
        AND SICDT_SICC_SEQ = SICC_SEQ;
   
BEGIN
    
   /* check formatting of the file name and path */
   IF UPPER(SUBSTR(RTRIM(p_file_name),-4,4)) != '.TXT' THEN
     RAISE ex_bad_file_ext;
   END IF;

   BEGIN
     v_file_handle := UTL_FILE.FOPEN(v_path,p_file_name, 'r');
   EXCEPTION
   	 WHEN OTHERS THEN
       RAISE ex_bad_file_handle;
   END;

   /* get import log sequence */
   p_import_log_seq := sim2_import_pkg.GET_IMPORT_LOG_SEQ;

   /* get org is number */
   OPEN c_get_org_is;
   FETCH c_get_org_is INTO v_org_is_nbr;
   CLOSE c_get_org_is;
   
   /* start import log record */
   BEGIN
     INSERT INTO SIM_IMPORT_LOG (
       SIL_SEQ,                
       SIL_SICC_SEQ,            
       SIL_IMPORT_ON,          
       SIL_IMPORTED_BY,        
       SIL_IMPORT_FILE,        
       SIL_LAB_ID,             
       SIL_ROWS_IMPORTED,      
       SIL_ROWS_ERRORS,        
       SIL_ROWS_READY,         
       SIL_DELETED_ON,         
       SIL_DELETED_BY,         
       SIL_EXPORTED_ON,        
       SIL_EXPORTED_BY,
       SIL_IMPORT_TYPE,
       SIL_TSMORGAN_IS_NUMBER,
       SIL_TSMORGAN_ORG_ID
       )
     VALUES (
       p_import_log_seq,       /* SIL_SEQ */
       p_imp_cfg_seq,          /* SIL_SICC_SEQ */ 
       SYSDATE,                /* SIL_IMPORT_ON */
       USER,                   /* SIL_IMPORTED_BY */
       p_file_name,            /* SIL_IMPORT_FILE */
       NULL,                   /* SIL_LAB_ID */
       NULL,                   /* SIL_ROWS_IMPORTED */
       NULL,                   /* SIL_ROWS_ERRORS */
       NULL,                   /* SIL_ROWS_READY */
       NULL,                   /* SIL_DELETED_ON */
       NULL,                   /* SIL_DELETED_BY */
       NULL,                   /* SIL_EXPORTED_ON */
       NULL,                   /* SIL_EXPORTED_BY */
       p_imp_type,             /* SIL_IMPORT_TYPE */
       v_org_is_nbr,           /* SIL_TSMORGAN_IS_NUMBER */
       p_org_id                /* SIL_TSMORGAN_ORG_ID */
       );

     COMMIT;

   END;

   BEGIN
     OPEN c_get_delimiter(p_imp_cfg_seq);
     FETCH c_get_delimiter INTO v_delimiter;
       IF c_get_delimiter%NOTFOUND THEN
         RAISE ex_bad_cfg;
       END IF;
     CLOSE c_get_delimiter;
   EXCEPTION
     WHEN OTHERS THEN
       RETURN SQLERRM||' - trying to get the delimiter';
   END;

   LOOP

      OPEN C_GET_NEXT_LINE_SEQ;
      FETCH C_GET_NEXT_LINE_SEQ INTO v_line_seq;
         IF C_GET_NEXT_LINE_SEQ%NOTFOUND THEN
            RAISE ex_no_sequence;
         END IF;
      CLOSE C_GET_NEXT_LINE_SEQ;
      
      v_line_cnt := v_line_cnt + 1;
       
      UTL_FILE.GET_LINE(v_file_handle,v_text_line);

      IF v_line_cnt = 1 THEN
        IF INSTR(v_text_line,v_delimiter) = 0 THEN
           RAISE ex_wrong_delimiter;
        END IF;
        
        /* count the number of fields in first line */
        v_no_fields := LENGTH(v_text_line) - LENGTH(TRANSLATE(v_text_line,'a'||v_delimiter,'a')) + 1;
        
        /* count the number of fields in import configuration */
        OPEN c_get_no_imp_fields(p_imp_cfg_seq);
        FETCH c_get_no_imp_fields INTO v_no_imp_fields;
          IF c_get_no_imp_fields%NOTFOUND THEN
          	RAISE ex_bad_cfg;
          END IF;
        CLOSE c_get_no_imp_fields;
        
        IF v_no_fields != v_no_imp_fields THEN
        	RAISE ex_wrong_no_fields;
        END IF;
      END IF;

      INSERT INTO sim_imp_lines(
         SIPL_SEQ,
         SIPL_SIL_SEQ,
         SIPL_LINE_TEXT,
         SIPL_LINE_NO,
         SIPL_STA_SEQ,
         SIPL_FA_SEQ,
         SIPL_RS_SEQ,
         SIPL_AC_SEQ,
         SIPL_SPJ_SEQ
         )
      VALUES(
         v_line_seq,                       /* SIPL_SEQ */
         p_import_log_seq,                 /* SIPL_SIL_SEQ */
         v_text_line,                      /* SIPL_LINE_TEXT */
         v_line_cnt,                       /* SIPL_LINE_NO */
         NULL,                             /* SIPL_STA_SEQ */
         NULL,                             /* SIPL_FA_SEQ */
         NULL,                             /* SIPL_RS_SEQ */
         NULL,                             /* SIPL_AC_SEQ */
         NULL                              /* SIPL_SPJ_SEQ */  
         );

      IF MOD(v_line_cnt,100) = 0 THEN
        COMMIT;
      END IF;
          
   END LOOP;   

   COMMIT;
   
   UTL_FILE.FCLOSE(v_file_handle);
   
   /* no errors, don't return anything */
   RETURN NULL;
   
EXCEPTION

   WHEN ex_bad_file_ext THEN
      RETURN 'Import file must be a text file and end with the .txt extension.';
   
   WHEN ex_bad_file_handle THEN
      RETURN 'Invalid file name or path.';

   WHEN ex_bad_cfg THEN
      RETURN 'Configuration is incomplete or invalid.';
   
   WHEN ex_wrong_delimiter THEN
      RETURN 'Delimiter could not be found in the text file. Make sure the file is in text format and the delimiter is right in the import configuration.';

   WHEN ex_wrong_no_fields THEN
      RETURN 'Import file contains the wrong number of fields, check the import configuration to make sure that it has been set up correctly.';

   WHEN ex_no_sequence THEN
      RETURN 'No sequence returned for sim_imp_lines.';

   WHEN NO_DATA_FOUND THEN
      UTL_FILE.FCLOSE(v_file_handle);
      COMMIT;
      RETURN NULL;
   
   WHEN OTHERS THEN 
      RETURN SQLERRM||' - in IMPF2_D.READ_TEXT_FILE function.';
 
END READ_TEXT_FILE;

/****************************************************************************************/

END sim2_web_txt_file_pkg;
/
